﻿using System.Runtime.InteropServices;
using System.Security;

namespace System.Data.SQLite.Vfs.Async
{
    // Class is fully thread safe. Initilize and Shutdown block all methods, multiple call of flush can be made at the same time.
    public static class SQLiteAsync
    {
        [SuppressUnmanagedCodeSecurity]
        private static class UnsafeNativeMethods
        {
            public const string SQLITEASYNC_VFSNAME = "sqlite3async";

            private const string SQLITE_DLL = "System.Data.SQLite.Vfs.Async.DLL";

            [DllImport(SQLITE_DLL)]
            public static extern int sqlite3async_initialize_noparent_interop(int isDefault);

            [DllImport(SQLITE_DLL)]
            public static extern int sqlite3async_start_interop();

            [DllImport(SQLITE_DLL)]
            public static extern int sqlite3async_flush_interop();

            [DllImport(SQLITE_DLL)]
            public static extern int sqlite3async_shutdown_interop();
        }

        private static System.Threading.ReaderWriterLock initilization_lock = new System.Threading.ReaderWriterLock();
        private static bool initialized;
        private static bool presentIsDefault;

        // TODO: For now only default mode is supported (e.g. apply to all connections)
        // To apply to a single connection SQLite would need to add support for specifying the vfs to use when 
        // creating a connection (final argument of sqlite3_open_v2() wich the sqlite interop always default to NULL at present)
        // I would imagine that crating connections to the same db with different vfs will cause issues
        // also need to take care no impact of connections being reused in the connection pool.

        public static void Initialize(bool isDefault)
        {
            try
            {
                initilization_lock.AcquireWriterLock(-1);

                if (!isDefault)
                    throw new NotSupportedException("isDefault=false - This needs support adding to system.data.sqlite to work");

                if (initialized)
                {
                    if (isDefault != presentIsDefault)
                        throw new InvalidOperationException("Already Initialized with a different default mode");
                    else
                        return;
                }

                int result = UnsafeNativeMethods.sqlite3async_initialize_noparent_interop(isDefault ? 1 : 0);
                if (result != 0)
                    throw new InvalidOperationException("Failed to initialize async vfs, error: " + result);

                result = UnsafeNativeMethods.sqlite3async_start_interop();

                if (result != 0)
                {
                    UnsafeNativeMethods.sqlite3async_shutdown_interop();
                    throw new InvalidOperationException("Failed to start worker thread, error: " + result);
                }

                presentIsDefault = isDefault;
                initialized = true;
//                System.Data.SQLite.SQLiteConnection.ClearAllPools();    // This might not be required, 
            }
            finally
            {
                initilization_lock.ReleaseWriterLock();
            }
        }

        public static void Flush()
        {
            try
            {
                initilization_lock.AcquireReaderLock(-1);

                if (!initialized)
                    return;

                int result = UnsafeNativeMethods.sqlite3async_flush_interop();

                if (result != 0)
                    throw new InvalidOperationException("Failed to flush, error: " + result);
            }
            finally
            {
                initilization_lock.ReleaseReaderLock();
            }
        }

        public static void Shutdown()
        {
            try
            {
                initilization_lock.AcquireWriterLock(-1);
            
                if (!initialized)
                    return;

                int result = UnsafeNativeMethods.sqlite3async_shutdown_interop();

                if (result != 0)
                    throw new InvalidOperationException("Failed to shutdown, error: " + result);

                initialized = false;
//                System.Data.SQLite.SQLiteConnection.ClearAllPools();
            }
            finally
            {
                initilization_lock.ReleaseWriterLock();
            }
        }
    }
}
