#include <windows.h>
#include "src/sqlite3.h"

typedef char *(*SQLITE3_VMPRINTF)(const char*, va_list);
typedef void *(*SQLITE3_MALLOC)(int);
typedef void (*SQLITE3_FREE)(void*);
typedef sqlite3_vfs *(*SQLITE3_VFS_FIND)(const char *);
typedef int (*SQLITE3_VFS_REGISTER)(sqlite3_vfs*, int);
typedef int (*SQLITE3_VFS_UNREGISTER)(sqlite3_vfs*);

SQLITE3_VMPRINTF sqlite3_vmprintf_dyn;
SQLITE3_MALLOC sqlite3_malloc_dyn;
SQLITE3_FREE sqlite3_free_dyn;
SQLITE3_VFS_FIND sqlite3_vfs_find_dyn;
SQLITE3_VFS_REGISTER sqlite3_vfs_register_dyn;
SQLITE3_VFS_UNREGISTER sqlite3_vfs_unregister_dyn;

int sqlite_refcount=0;
HMODULE sqlite_dll;

BOOL UnLoadSqliteDll()
{
	BOOL result = TRUE;

	if(--sqlite_refcount==0)
	{
		sqlite3_vmprintf_dyn = NULL;
		sqlite3_malloc_dyn = NULL;
		sqlite3_free_dyn = NULL;
		sqlite3_vfs_find_dyn = NULL;
		sqlite3_vfs_unregister_dyn = NULL;
		result = FreeLibrary(sqlite_dll);
		sqlite_dll = NULL;
	}

	return result;
}

BOOL LoadSqliteDll()
{
    if(++sqlite_refcount>1)
        return TRUE;

    if(sqlite_dll==NULL)
        sqlite_dll = LoadLibrary(L"System.Data.SQLite.DLL");
    
    if(sqlite_dll==NULL)
	{
		sqlite_refcount = 0;
		return FALSE;
	}

    sqlite3_vmprintf_dyn = (SQLITE3_VMPRINTF) GetProcAddress(sqlite_dll, "sqlite3_vmprintf");
    sqlite3_malloc_dyn = (SQLITE3_MALLOC) GetProcAddress(sqlite_dll, "sqlite3_malloc");
    sqlite3_free_dyn = (SQLITE3_FREE) GetProcAddress(sqlite_dll, "sqlite3_free");
    sqlite3_vfs_find_dyn = (SQLITE3_VFS_FIND) GetProcAddress(sqlite_dll, "sqlite3_vfs_find");
    sqlite3_vfs_register_dyn = (SQLITE3_VFS_REGISTER) GetProcAddress(sqlite_dll, "sqlite3_vfs_register");
    sqlite3_vfs_unregister_dyn = (SQLITE3_VFS_UNREGISTER) GetProcAddress(sqlite_dll, "sqlite3_vfs_unregister");
    
    if( sqlite3_vmprintf_dyn == NULL 
						|| sqlite3_malloc_dyn == NULL
						|| sqlite3_free_dyn == NULL
						|| sqlite3_vfs_find_dyn == NULL
						|| sqlite3_vfs_register_dyn == NULL
						|| sqlite3_vfs_unregister_dyn == NULL)
	{
		UnLoadSqliteDll();
		return FALSE;
	}
	else
	{
		return TRUE;
	}
}

// Redirect all calls to our dynamically loaded versions.
#define sqlite3_vmprintf sqlite3_vmprintf_dyn
#define sqlite3_malloc sqlite3_malloc_dyn
#define sqlite3_free sqlite3_free_dyn
#define sqlite3_vfs_find sqlite3_vfs_find_dyn
#define sqlite3_vfs_register sqlite3_vfs_register_dyn
#define sqlite3_vfs_unregister sqlite3_vfs_unregister_dyn

#define SQLITE_ENABLE_ASYNCIO 1
#include "src/sqlite3async.c"

BOOL async_worker_running=FALSE;
HANDLE async_worker_thread=NULL;

DWORD WINAPI worker(LPVOID args)
{
	// In a loop to save having to restart the thread when we perform a flush
	while(async_worker_running == TRUE)
	{
		sqlite3async_run();
		Sleep(0);	// ensure the thread performing the flush get a chance to grab the mutexes for its use of sqlite3async_run()
	}

	return 0;
}

__declspec(dllexport) int WINAPI sqlite3async_initialize_interop(const char *zParent, int isDefault)
{
	if(LoadSqliteDll()==FALSE)
		return GetLastError();

	return sqlite3async_initialize(zParent, isDefault);
}

__declspec(dllexport) int WINAPI sqlite3async_initialize_noparent_interop(int isDefault)
{
	if(LoadSqliteDll()==FALSE)
		return GetLastError();

	return sqlite3async_initialize(NULL, isDefault);
}

__declspec(dllexport) int WINAPI sqlite3async_start_interop()
{
	if(async_worker_running==TRUE)
		return SQLITE_OK;
	
	async_worker_thread = CreateThread(NULL, 0, worker, NULL, 0, NULL);

	if(async_worker_thread!=NULL)
	{
		async_worker_running = TRUE;
		return SQLITE_OK;
	}
	else
	{
		return GetLastError();
	}
}

__declspec(dllexport) int WINAPI sqlite3async_flush_interop()
{
	int result = sqlite3async_control(SQLITEASYNC_HALT, SQLITEASYNC_HALT_IDLE);
	if(result!=SQLITE_OK)
		return result;

	sqlite3async_run();

	return sqlite3async_control(SQLITEASYNC_HALT, SQLITEASYNC_HALT_NEVER);
}

__declspec(dllexport) int WINAPI sqlite3async_shutdown_interop()
{
	int result = SQLITE_OK;

	if(async_worker_running==TRUE)
	{
		// Make the thread exit when it has no more work to perform
		async_worker_running = FALSE;
		result = sqlite3async_control(SQLITEASYNC_HALT, SQLITEASYNC_HALT_IDLE);

		if(result!=SQLITE_OK)
		{
			async_worker_running = TRUE;
			return result;
		}

		WaitForSingleObject(async_worker_thread, INFINITE);
	}

	sqlite3async_shutdown();
	UnLoadSqliteDll();

	return result;
}